from flask_sqlalchemy import SQLAlchemy
from flask_login import UserMixin
from werkzeug.security import generate_password_hash, check_password_hash

db = SQLAlchemy()

class User(db.Model, UserMixin):
    id = db.Column(db.Integer, primary_key=True)
    nome = db.Column(db.String(100), nullable=False)
    email = db.Column(db.String(120), unique=True, nullable=False)
    senha_hash = db.Column(db.String(256), nullable=False)
    tipo_perfil = db.Column(db.String(20), nullable=False)  # 'administrador', 'consultor', 'cliente'
    ativo = db.Column(db.Boolean, default=True)
    empresa_id = db.Column(db.Integer, db.ForeignKey('empresa.id'), nullable=True)
    
    def __init__(self, nome, email, senha, tipo_perfil, empresa_id=None):
        self.nome = nome
        self.email = email
        self.senha_hash = generate_password_hash(senha)
        self.tipo_perfil = tipo_perfil
        self.empresa_id = empresa_id
        self.ativo = True
    
    def set_password(self, senha):
        self.senha_hash = generate_password_hash(senha)
    
    def check_password(self, senha):
        return check_password_hash(self.senha_hash, senha)
    
    def is_active(self):
        return self.ativo
    
    def __repr__(self):
        return f'<User {self.nome}>'
    
    def to_dict(self):
        return {
            'id': self.id,
            'nome': self.nome,
            'email': self.email,
            'tipo_perfil': self.tipo_perfil,
            'ativo': self.ativo,
            'empresa_id': self.empresa_id
        }

class Empresa(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    razao_social = db.Column(db.String(200), nullable=False)
    nome_fantasia = db.Column(db.String(200), nullable=False)
    cnpj = db.Column(db.String(18), unique=True, nullable=False)
    segmento_id = db.Column(db.Integer, db.ForeignKey('segmento.id'), nullable=False)
    cidade = db.Column(db.String(100), nullable=False)
    estado = db.Column(db.String(2), nullable=False)
    faixa_faturamento_id = db.Column(db.Integer, db.ForeignKey('faixa_faturamento.id'), nullable=False)
    data_abertura = db.Column(db.Date, nullable=False)
    ativo = db.Column(db.Boolean, default=True)
    
    usuarios = db.relationship('User', backref='empresa', lazy=True)
    
    def to_dict(self):
        return {
            'id': self.id,
            'razao_social': self.razao_social,
            'nome_fantasia': self.nome_fantasia,
            'cnpj': self.cnpj,
            'segmento_id': self.segmento_id,
            'cidade': self.cidade,
            'estado': self.estado,
            'faixa_faturamento_id': self.faixa_faturamento_id,
            'data_abertura': self.data_abertura.strftime('%Y-%m-%d'),
            'ativo': self.ativo
        }

class Segmento(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    codigo = db.Column(db.String(20), nullable=False)
    nome = db.Column(db.String(100), nullable=False)
    ativo = db.Column(db.Boolean, default=True)
    
    empresas = db.relationship('Empresa', backref='segmento', lazy=True)
    
    def to_dict(self):
        return {
            'id': self.id,
            'codigo': self.codigo,
            'nome': self.nome,
            'ativo': self.ativo
        }

class FaixaFaturamento(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    codigo = db.Column(db.String(20), nullable=False)
    valor_de = db.Column(db.Float, nullable=False)
    valor_ate = db.Column(db.Float, nullable=False)
    descricao = db.Column(db.String(100), nullable=False)
    ativo = db.Column(db.Boolean, default=True)
    
    empresas = db.relationship('Empresa', backref='faixa_faturamento', lazy=True)
    
    def to_dict(self):
        return {
            'id': self.id,
            'codigo': self.codigo,
            'valor_de': self.valor_de,
            'valor_ate': self.valor_ate,
            'descricao': self.descricao,
            'ativo': self.ativo
        }
