from flask import Blueprint, render_template, request, jsonify, flash, redirect, url_for
from flask_login import login_required, current_user
from src.models.user import db, Atividade

admin_atividade_bp = Blueprint('admin_atividade', __name__)

@admin_atividade_bp.route('/atividades')
@login_required
def listar_atividades():
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    atividades = Atividade.query.all()
    return render_template('admin/atividades/listar.html', atividades=atividades)

@admin_atividade_bp.route('/atividades/nova', methods=['GET', 'POST'])
@login_required
def nova_atividade():
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    if request.method == 'POST':
        codigo = request.form.get('codigo')
        descricao = request.form.get('descricao')
        ativo = True if request.form.get('ativo') == 'on' else False
        
        # Validar dados
        if not all([codigo, descricao]):
            flash('Por favor, preencha todos os campos obrigatórios.', 'danger')
            return render_template('admin/atividades/form.html')
        
        # Criar nova atividade
        atividade = Atividade(
            codigo=codigo,
            descricao=descricao,
            ativo=ativo
        )
        
        db.session.add(atividade)
        db.session.commit()
        
        flash('Atividade cadastrada com sucesso!', 'success')
        return redirect(url_for('admin_atividade.listar_atividades'))
    
    return render_template('admin/atividades/form.html')

@admin_atividade_bp.route('/atividades/editar/<int:atividade_id>', methods=['GET', 'POST'])
@login_required
def editar_atividade(atividade_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    atividade = Atividade.query.get_or_404(atividade_id)
    
    if request.method == 'POST':
        codigo = request.form.get('codigo')
        descricao = request.form.get('descricao')
        ativo = True if request.form.get('ativo') == 'on' else False
        
        # Validar dados
        if not all([codigo, descricao]):
            flash('Por favor, preencha todos os campos obrigatórios.', 'danger')
            return render_template('admin/atividades/form.html', atividade=atividade)
        
        # Atualizar atividade
        atividade.codigo = codigo
        atividade.descricao = descricao
        atividade.ativo = ativo
        
        db.session.commit()
        
        flash('Atividade atualizada com sucesso!', 'success')
        return redirect(url_for('admin_atividade.listar_atividades'))
    
    return render_template('admin/atividades/form.html', atividade=atividade)

@admin_atividade_bp.route('/atividades/excluir/<int:atividade_id>', methods=['POST'])
@login_required
def excluir_atividade(atividade_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return jsonify({'success': False, 'message': 'Acesso não autorizado'}), 403
    
    atividade = Atividade.query.get_or_404(atividade_id)
    
    # Verificar se a atividade está sendo usada por alguma pergunta
    # Isso depende da implementação do modelo de Pergunta
    # if atividade.perguntas:
    #     return jsonify({'success': False, 'message': 'Esta atividade está sendo utilizada por perguntas e não pode ser excluída.'}), 400
    
    db.session.delete(atividade)
    db.session.commit()
    
    return jsonify({'success': True, 'message': 'Atividade excluída com sucesso!'})

@admin_atividade_bp.route('/atividades/alternar-status/<int:atividade_id>', methods=['POST'])
@login_required
def alternar_status(atividade_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return jsonify({'success': False, 'message': 'Acesso não autorizado'}), 403
    
    atividade = Atividade.query.get_or_404(atividade_id)
    atividade.ativo = not atividade.ativo
    db.session.commit()
    
    status = 'ativada' if atividade.ativo else 'desativada'
    return jsonify({'success': True, 'message': f'Atividade {status} com sucesso!', 'ativo': atividade.ativo})
