from flask import Blueprint, render_template, request, jsonify, flash, redirect, url_for
from flask_login import login_required, current_user
from src.models.user import db, Pergunta, Assunto, Setor, Atividade

admin_pergunta_bp = Blueprint('admin_pergunta', __name__)

@admin_pergunta_bp.route('/perguntas')
@login_required
def listar_perguntas():
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    perguntas = Pergunta.query.all()
    return render_template('admin/perguntas/listar.html', perguntas=perguntas)

@admin_pergunta_bp.route('/perguntas/nova', methods=['GET', 'POST'])
@login_required
def nova_pergunta():
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    assuntos = Assunto.query.filter_by(ativo=True).all()
    setores = Setor.query.filter_by(ativo=True).all()
    atividades = Atividade.query.filter_by(ativo=True).all()
    
    if request.method == 'POST':
        codigo = request.form.get('codigo')
        texto = request.form.get('texto')
        tipo_resposta = request.form.get('tipo_resposta')
        assunto_id = request.form.get('assunto_id')
        setor_id = request.form.get('setor_id')
        atividade_id = request.form.get('atividade_id')
        opcoes_resposta = request.form.get('opcoes_resposta')
        ativo = True if request.form.get('ativo') == 'on' else False
        
        # Validar dados
        if not all([codigo, texto, tipo_resposta, assunto_id, setor_id, atividade_id]):
            flash('Por favor, preencha todos os campos obrigatórios.', 'danger')
            return render_template('admin/perguntas/form.html', assuntos=assuntos, setores=setores, atividades=atividades)
        
        # Verificar se o tipo de resposta é válido
        if tipo_resposta not in ['texto', 'numero', 'selecao', 'multipla_escolha']:
            flash('Tipo de resposta inválido.', 'danger')
            return render_template('admin/perguntas/form.html', assuntos=assuntos, setores=setores, atividades=atividades)
        
        # Se for tipo seleção ou múltipla escolha, opções de resposta são obrigatórias
        if tipo_resposta in ['selecao', 'multipla_escolha'] and not opcoes_resposta:
            flash('Para perguntas de seleção ou múltipla escolha, é necessário fornecer opções de resposta.', 'danger')
            return render_template('admin/perguntas/form.html', assuntos=assuntos, setores=setores, atividades=atividades)
        
        # Criar nova pergunta
        pergunta = Pergunta(
            codigo=codigo,
            texto=texto,
            tipo_resposta=tipo_resposta,
            assunto_id=assunto_id,
            setor_id=setor_id,
            atividade_id=atividade_id,
            opcoes_resposta=opcoes_resposta,
            ativo=ativo
        )
        
        db.session.add(pergunta)
        db.session.commit()
        
        flash('Pergunta cadastrada com sucesso!', 'success')
        return redirect(url_for('admin_pergunta.listar_perguntas'))
    
    return render_template('admin/perguntas/form.html', assuntos=assuntos, setores=setores, atividades=atividades)

@admin_pergunta_bp.route('/perguntas/editar/<int:pergunta_id>', methods=['GET', 'POST'])
@login_required
def editar_pergunta(pergunta_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    pergunta = Pergunta.query.get_or_404(pergunta_id)
    assuntos = Assunto.query.filter_by(ativo=True).all()
    setores = Setor.query.filter_by(ativo=True).all()
    atividades = Atividade.query.filter_by(ativo=True).all()
    
    if request.method == 'POST':
        codigo = request.form.get('codigo')
        texto = request.form.get('texto')
        tipo_resposta = request.form.get('tipo_resposta')
        assunto_id = request.form.get('assunto_id')
        setor_id = request.form.get('setor_id')
        atividade_id = request.form.get('atividade_id')
        opcoes_resposta = request.form.get('opcoes_resposta')
        ativo = True if request.form.get('ativo') == 'on' else False
        
        # Validar dados
        if not all([codigo, texto, tipo_resposta, assunto_id, setor_id, atividade_id]):
            flash('Por favor, preencha todos os campos obrigatórios.', 'danger')
            return render_template('admin/perguntas/form.html', pergunta=pergunta, assuntos=assuntos, setores=setores, atividades=atividades)
        
        # Verificar se o tipo de resposta é válido
        if tipo_resposta not in ['texto', 'numero', 'selecao', 'multipla_escolha']:
            flash('Tipo de resposta inválido.', 'danger')
            return render_template('admin/perguntas/form.html', pergunta=pergunta, assuntos=assuntos, setores=setores, atividades=atividades)
        
        # Se for tipo seleção ou múltipla escolha, opções de resposta são obrigatórias
        if tipo_resposta in ['selecao', 'multipla_escolha'] and not opcoes_resposta:
            flash('Para perguntas de seleção ou múltipla escolha, é necessário fornecer opções de resposta.', 'danger')
            return render_template('admin/perguntas/form.html', pergunta=pergunta, assuntos=assuntos, setores=setores, atividades=atividades)
        
        # Atualizar pergunta
        pergunta.codigo = codigo
        pergunta.texto = texto
        pergunta.tipo_resposta = tipo_resposta
        pergunta.assunto_id = assunto_id
        pergunta.setor_id = setor_id
        pergunta.atividade_id = atividade_id
        pergunta.opcoes_resposta = opcoes_resposta
        pergunta.ativo = ativo
        
        db.session.commit()
        
        flash('Pergunta atualizada com sucesso!', 'success')
        return redirect(url_for('admin_pergunta.listar_perguntas'))
    
    return render_template('admin/perguntas/form.html', pergunta=pergunta, assuntos=assuntos, setores=setores, atividades=atividades)

@admin_pergunta_bp.route('/perguntas/excluir/<int:pergunta_id>', methods=['POST'])
@login_required
def excluir_pergunta(pergunta_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return jsonify({'success': False, 'message': 'Acesso não autorizado'}), 403
    
    pergunta = Pergunta.query.get_or_404(pergunta_id)
    
    # Verificar se a pergunta possui respostas
    # Isso depende da implementação do modelo de Resposta
    # if pergunta.respostas:
    #     return jsonify({'success': False, 'message': 'Esta pergunta possui respostas e não pode ser excluída.'}), 400
    
    db.session.delete(pergunta)
    db.session.commit()
    
    return jsonify({'success': True, 'message': 'Pergunta excluída com sucesso!'})

@admin_pergunta_bp.route('/perguntas/alternar-status/<int:pergunta_id>', methods=['POST'])
@login_required
def alternar_status(pergunta_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return jsonify({'success': False, 'message': 'Acesso não autorizado'}), 403
    
    pergunta = Pergunta.query.get_or_404(pergunta_id)
    pergunta.ativo = not pergunta.ativo
    db.session.commit()
    
    status = 'ativada' if pergunta.ativo else 'desativada'
    return jsonify({'success': True, 'message': f'Pergunta {status} com sucesso!', 'ativo': pergunta.ativo})
