from flask import Blueprint, render_template, request, jsonify, flash, redirect, url_for
from flask_login import login_required, current_user
from src.models.user import db, Setor

admin_setor_bp = Blueprint('admin_setor', __name__)

@admin_setor_bp.route('/setores')
@login_required
def listar_setores():
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    setores = Setor.query.all()
    return render_template('admin/setores/listar.html', setores=setores)

@admin_setor_bp.route('/setores/novo', methods=['GET', 'POST'])
@login_required
def novo_setor():
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    if request.method == 'POST':
        codigo = request.form.get('codigo')
        descricao = request.form.get('descricao')
        ativo = True if request.form.get('ativo') == 'on' else False
        
        # Validar dados
        if not all([codigo, descricao]):
            flash('Por favor, preencha todos os campos obrigatórios.', 'danger')
            return render_template('admin/setores/form.html')
        
        # Criar novo setor
        setor = Setor(
            codigo=codigo,
            descricao=descricao,
            ativo=ativo
        )
        
        db.session.add(setor)
        db.session.commit()
        
        flash('Setor cadastrado com sucesso!', 'success')
        return redirect(url_for('admin_setor.listar_setores'))
    
    return render_template('admin/setores/form.html')

@admin_setor_bp.route('/setores/editar/<int:setor_id>', methods=['GET', 'POST'])
@login_required
def editar_setor(setor_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    setor = Setor.query.get_or_404(setor_id)
    
    if request.method == 'POST':
        codigo = request.form.get('codigo')
        descricao = request.form.get('descricao')
        ativo = True if request.form.get('ativo') == 'on' else False
        
        # Validar dados
        if not all([codigo, descricao]):
            flash('Por favor, preencha todos os campos obrigatórios.', 'danger')
            return render_template('admin/setores/form.html', setor=setor)
        
        # Atualizar setor
        setor.codigo = codigo
        setor.descricao = descricao
        setor.ativo = ativo
        
        db.session.commit()
        
        flash('Setor atualizado com sucesso!', 'success')
        return redirect(url_for('admin_setor.listar_setores'))
    
    return render_template('admin/setores/form.html', setor=setor)

@admin_setor_bp.route('/setores/excluir/<int:setor_id>', methods=['POST'])
@login_required
def excluir_setor(setor_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return jsonify({'success': False, 'message': 'Acesso não autorizado'}), 403
    
    setor = Setor.query.get_or_404(setor_id)
    
    # Verificar se o setor está sendo usado por alguma pergunta
    # Isso depende da implementação do modelo de Pergunta
    # if setor.perguntas:
    #     return jsonify({'success': False, 'message': 'Este setor está sendo utilizado por perguntas e não pode ser excluído.'}), 400
    
    db.session.delete(setor)
    db.session.commit()
    
    return jsonify({'success': True, 'message': 'Setor excluído com sucesso!'})

@admin_setor_bp.route('/setores/alternar-status/<int:setor_id>', methods=['POST'])
@login_required
def alternar_status(setor_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return jsonify({'success': False, 'message': 'Acesso não autorizado'}), 403
    
    setor = Setor.query.get_or_404(setor_id)
    setor.ativo = not setor.ativo
    db.session.commit()
    
    status = 'ativado' if setor.ativo else 'desativado'
    return jsonify({'success': True, 'message': f'Setor {status} com sucesso!', 'ativo': setor.ativo})
