from flask import Blueprint, render_template, redirect, url_for, flash, request, jsonify
from flask_login import login_user, logout_user, login_required, current_user
from src.models.user import User, db
from werkzeug.security import generate_password_hash

auth_bp = Blueprint('auth', __name__)
user_bp = Blueprint('user', __name__)

# Rotas de autenticação
@auth_bp.route('/login', methods=['GET', 'POST'])
def login():
    if request.method == 'POST':
        email = request.form.get('email')
        senha = request.form.get('senha')
        
        user = User.query.filter_by(email=email).first()
        
        if user and user.check_password(senha) and user.ativo:
            login_user(user)
            
            # Redirecionar para o dashboard apropriado com base no tipo de perfil
            if user.tipo_perfil == 'administrador':
                return redirect(url_for('admin.dashboard'))
            elif user.tipo_perfil == 'consultor':
                return redirect(url_for('consultor.dashboard'))
            else:  # cliente
                return redirect(url_for('cliente.dashboard'))
        
        flash('Credenciais inválidas ou usuário inativo.')
        return redirect(url_for('auth.login'))
    
    return render_template('auth/login.html')

@auth_bp.route('/logout')
@login_required
def logout():
    logout_user()
    flash('Você foi desconectado com sucesso.')
    return redirect(url_for('auth.login'))

@auth_bp.route('/recuperar-senha', methods=['GET', 'POST'])
def recuperar_senha():
    if request.method == 'POST':
        email = request.form.get('email')
        user = User.query.filter_by(email=email).first()
        
        if user:
            # Em um sistema real, enviaríamos um e-mail com um token
            # Para simplificar, vamos apenas redefinir a senha para um valor padrão
            nova_senha = 'senha123'  # Em produção, seria um valor aleatório
            user.set_password(nova_senha)
            db.session.commit()
            
            flash(f'Sua senha foi redefinida para: {nova_senha}')
            return redirect(url_for('auth.login'))
        
        flash('E-mail não encontrado.')
        return redirect(url_for('auth.recuperar_senha'))
    
    return render_template('auth/recuperar_senha.html')

# Rotas de usuário (API)
@user_bp.route('/users', methods=['GET'])
@login_required
def get_users():
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return jsonify({'error': 'Acesso não autorizado'}), 403
    
    users = User.query.all()
    return jsonify([user.to_dict() for user in users])

@user_bp.route('/users', methods=['POST'])
@login_required
def create_user():
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return jsonify({'error': 'Acesso não autorizado'}), 403
    
    data = request.json
    
    # Validar dados
    if not all(k in data for k in ('nome', 'email', 'senha', 'tipo_perfil')):
        return jsonify({'error': 'Dados incompletos'}), 400
    
    # Verificar se o e-mail já existe
    if User.query.filter_by(email=data['email']).first():
        return jsonify({'error': 'E-mail já cadastrado'}), 400
    
    # Criar novo usuário
    user = User(
        nome=data['nome'],
        email=data['email'],
        senha=data['senha'],
        tipo_perfil=data['tipo_perfil'],
        empresa_id=data.get('empresa_id')
    )
    
    db.session.add(user)
    db.session.commit()
    
    return jsonify(user.to_dict()), 201

@user_bp.route('/users/<int:user_id>', methods=['GET'])
@login_required
def get_user(user_id):
    # Verificar se o usuário é administrador ou é o próprio usuário
    if current_user.tipo_perfil != 'administrador' and current_user.id != user_id:
        return jsonify({'error': 'Acesso não autorizado'}), 403
    
    user = User.query.get_or_404(user_id)
    return jsonify(user.to_dict())

@user_bp.route('/users/<int:user_id>', methods=['PUT'])
@login_required
def update_user(user_id):
    # Verificar se o usuário é administrador ou é o próprio usuário
    if current_user.tipo_perfil != 'administrador' and current_user.id != user_id:
        return jsonify({'error': 'Acesso não autorizado'}), 403
    
    user = User.query.get_or_404(user_id)
    data = request.json
    
    # Atualizar campos
    if 'nome' in data:
        user.nome = data['nome']
    if 'email' in data and data['email'] != user.email:
        # Verificar se o novo e-mail já existe
        if User.query.filter_by(email=data['email']).first():
            return jsonify({'error': 'E-mail já cadastrado'}), 400
        user.email = data['email']
    if 'senha' in data:
        user.set_password(data['senha'])
    if 'tipo_perfil' in data and current_user.tipo_perfil == 'administrador':
        user.tipo_perfil = data['tipo_perfil']
    if 'ativo' in data and current_user.tipo_perfil == 'administrador':
        user.ativo = data['ativo']
    if 'empresa_id' in data and current_user.tipo_perfil == 'administrador':
        user.empresa_id = data['empresa_id']
    
    db.session.commit()
    
    return jsonify(user.to_dict())

@user_bp.route('/users/<int:user_id>', methods=['DELETE'])
@login_required
def delete_user(user_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return jsonify({'error': 'Acesso não autorizado'}), 403
    
    user = User.query.get_or_404(user_id)
    
    # Não permitir que o administrador exclua a si mesmo
    if user.id == current_user.id:
        return jsonify({'error': 'Não é possível excluir o próprio usuário'}), 400
    
    db.session.delete(user)
    db.session.commit()
    
    return '', 204
