#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os
import sys
from flask_sqlalchemy import SQLAlchemy
from werkzeug.security import generate_password_hash
from datetime import datetime

# Adicionar o diretório pai ao path para importar os módulos
sys.path.insert(0, os.path.dirname(os.path.dirname(__file__)))

# Importar a aplicação e os modelos
from src.main import app, db
from src.models.user import User, Empresa, FaixaFaturamento, Segmento, Setor, Assunto, Atividade, Pergunta

def init_db():
    """Inicializa o banco de dados com dados básicos"""
    with app.app_context():
        # Criar todas as tabelas
        db.create_all()
        
        # Verificar se já existem dados
        if User.query.first() is not None:
            print("O banco de dados já foi inicializado.")
            return
        
        print("Inicializando banco de dados...")
        
        # Criar usuário administrador padrão
        admin = User(
            nome="Administrador",
            email="admin@vinci.com",
            senha_hash=generate_password_hash("admin123"),
            tipo_perfil="administrador",
            ativo=True
        )
        db.session.add(admin)
        
        # Criar faixas de faturamento
        faixas = [
            FaixaFaturamento(descricao="Até R$ 500 mil", ativo=True),
            FaixaFaturamento(descricao="De R$ 500 mil a R$ 2 milhões", ativo=True),
            FaixaFaturamento(descricao="De R$ 2 milhões a R$ 10 milhões", ativo=True),
            FaixaFaturamento(descricao="De R$ 10 milhões a R$ 50 milhões", ativo=True),
            FaixaFaturamento(descricao="Acima de R$ 50 milhões", ativo=True)
        ]
        db.session.add_all(faixas)
        
        # Criar segmentos
        segmentos = [
            Segmento(nome="Indústria", ativo=True),
            Segmento(nome="Comércio", ativo=True),
            Segmento(nome="Serviços", ativo=True),
            Segmento(nome="Tecnologia", ativo=True),
            Segmento(nome="Agronegócio", ativo=True)
        ]
        db.session.add_all(segmentos)
        
        # Criar setores
        setores = [
            Setor(descricao="Financeiro", ativo=True),
            Setor(descricao="Recursos Humanos", ativo=True),
            Setor(descricao="Produção", ativo=True),
            Setor(descricao="Marketing", ativo=True),
            Setor(descricao="Vendas", ativo=True),
            Setor(descricao="Logística", ativo=True)
        ]
        db.session.add_all(setores)
        
        # Criar assuntos
        assuntos = [
            Assunto(descricao="Gestão Financeira", ativo=True),
            Assunto(descricao="Gestão de Pessoas", ativo=True),
            Assunto(descricao="Processos Produtivos", ativo=True),
            Assunto(descricao="Estratégias de Marketing", ativo=True),
            Assunto(descricao="Técnicas de Vendas", ativo=True),
            Assunto(descricao="Gestão de Estoque", ativo=True)
        ]
        db.session.add_all(assuntos)
        
        # Criar atividades
        atividades = [
            Atividade(descricao="Planejamento", ativo=True),
            Atividade(descricao="Execução", ativo=True),
            Atividade(descricao="Controle", ativo=True),
            Atividade(descricao="Análise", ativo=True)
        ]
        db.session.add_all(atividades)
        
        # Commit para salvar os dados e obter os IDs
        db.session.commit()
        
        # Criar empresas de exemplo
        empresas = [
            Empresa(
                razao_social="Empresa Exemplo 1 Ltda",
                nome_fantasia="Empresa 1",
                cnpj="12.345.678/0001-01",
                segmento_id=1,
                faixa_faturamento_id=2,
                cidade="São Paulo",
                estado="SP",
                data_abertura=datetime.strptime("2010-01-01", "%Y-%m-%d"),
                ativo=True
            ),
            Empresa(
                razao_social="Empresa Exemplo 2 Ltda",
                nome_fantasia="Empresa 2",
                cnpj="23.456.789/0001-02",
                segmento_id=2,
                faixa_faturamento_id=3,
                cidade="Rio de Janeiro",
                estado="RJ",
                data_abertura=datetime.strptime("2012-05-10", "%Y-%m-%d"),
                ativo=True
            )
        ]
        db.session.add_all(empresas)
        
        # Criar usuários de exemplo
        usuarios = [
            User(
                nome="Consultor Exemplo",
                email="consultor@vinci.com",
                senha_hash=generate_password_hash("consultor123"),
                tipo_perfil="consultor",
                ativo=True
            ),
            User(
                nome="Cliente Exemplo",
                email="cliente@vinci.com",
                senha_hash=generate_password_hash("cliente123"),
                tipo_perfil="cliente",
                empresa_id=1,
                ativo=True
            )
        ]
        db.session.add_all(usuarios)
        
        # Commit para salvar os dados e obter os IDs
        db.session.commit()
        
        # Criar perguntas de exemplo
        perguntas = [
            Pergunta(
                codigo="FIN001",
                texto="Qual o faturamento anual da empresa?",
                tipo_resposta="numero",
                assunto_id=1,
                setor_id=1,
                atividade_id=4,
                ativo=True
            ),
            Pergunta(
                codigo="FIN002",
                texto="Qual o percentual de lucro líquido sobre o faturamento?",
                tipo_resposta="numero",
                assunto_id=1,
                setor_id=1,
                atividade_id=4,
                ativo=True
            ),
            Pergunta(
                codigo="RH001",
                texto="Quantos funcionários a empresa possui?",
                tipo_resposta="numero",
                assunto_id=2,
                setor_id=2,
                atividade_id=4,
                ativo=True
            ),
            Pergunta(
                codigo="MKT001",
                texto="Quais canais de marketing são utilizados?",
                tipo_resposta="multipla_escolha",
                opcoes_resposta="Redes Sociais;Email Marketing;Google Ads;Outdoor;TV;Rádio",
                assunto_id=4,
                setor_id=4,
                atividade_id=1,
                ativo=True
            ),
            Pergunta(
                codigo="VND001",
                texto="Qual o principal canal de vendas?",
                tipo_resposta="selecao",
                opcoes_resposta="Loja Física;E-commerce;Televendas;Representantes;Distribuidores",
                assunto_id=5,
                setor_id=5,
                atividade_id=2,
                ativo=True
            )
        ]
        db.session.add_all(perguntas)
        
        # Commit final
        db.session.commit()
        
        print("Banco de dados inicializado com sucesso!")

if __name__ == "__main__":
    init_db()
