from flask import Blueprint, render_template, request, jsonify, flash, redirect, url_for
from flask_login import login_required, current_user
from src.models.user import db, Assunto

admin_assunto_bp = Blueprint('admin_assunto', __name__)

@admin_assunto_bp.route('/assuntos')
@login_required
def listar_assuntos():
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    assuntos = Assunto.query.all()
    return render_template('admin/assuntos/listar.html', assuntos=assuntos)

@admin_assunto_bp.route('/assuntos/novo', methods=['GET', 'POST'])
@login_required
def novo_assunto():
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    if request.method == 'POST':
        codigo = request.form.get('codigo')
        descricao = request.form.get('descricao')
        ativo = True if request.form.get('ativo') == 'on' else False
        
        # Validar dados
        if not all([codigo, descricao]):
            flash('Por favor, preencha todos os campos obrigatórios.', 'danger')
            return render_template('admin/assuntos/form.html')
        
        # Criar novo assunto
        assunto = Assunto(
            codigo=codigo,
            descricao=descricao,
            ativo=ativo
        )
        
        db.session.add(assunto)
        db.session.commit()
        
        flash('Assunto cadastrado com sucesso!', 'success')
        return redirect(url_for('admin_assunto.listar_assuntos'))
    
    return render_template('admin/assuntos/form.html')

@admin_assunto_bp.route('/assuntos/editar/<int:assunto_id>', methods=['GET', 'POST'])
@login_required
def editar_assunto(assunto_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    assunto = Assunto.query.get_or_404(assunto_id)
    
    if request.method == 'POST':
        codigo = request.form.get('codigo')
        descricao = request.form.get('descricao')
        ativo = True if request.form.get('ativo') == 'on' else False
        
        # Validar dados
        if not all([codigo, descricao]):
            flash('Por favor, preencha todos os campos obrigatórios.', 'danger')
            return render_template('admin/assuntos/form.html', assunto=assunto)
        
        # Atualizar assunto
        assunto.codigo = codigo
        assunto.descricao = descricao
        assunto.ativo = ativo
        
        db.session.commit()
        
        flash('Assunto atualizado com sucesso!', 'success')
        return redirect(url_for('admin_assunto.listar_assuntos'))
    
    return render_template('admin/assuntos/form.html', assunto=assunto)

@admin_assunto_bp.route('/assuntos/excluir/<int:assunto_id>', methods=['POST'])
@login_required
def excluir_assunto(assunto_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return jsonify({'success': False, 'message': 'Acesso não autorizado'}), 403
    
    assunto = Assunto.query.get_or_404(assunto_id)
    
    # Verificar se o assunto está sendo usado por alguma pergunta
    # Isso depende da implementação do modelo de Pergunta
    # if assunto.perguntas:
    #     return jsonify({'success': False, 'message': 'Este assunto está sendo utilizado por perguntas e não pode ser excluído.'}), 400
    
    db.session.delete(assunto)
    db.session.commit()
    
    return jsonify({'success': True, 'message': 'Assunto excluído com sucesso!'})

@admin_assunto_bp.route('/assuntos/alternar-status/<int:assunto_id>', methods=['POST'])
@login_required
def alternar_status(assunto_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return jsonify({'success': False, 'message': 'Acesso não autorizado'}), 403
    
    assunto = Assunto.query.get_or_404(assunto_id)
    assunto.ativo = not assunto.ativo
    db.session.commit()
    
    status = 'ativado' if assunto.ativo else 'desativado'
    return jsonify({'success': True, 'message': f'Assunto {status} com sucesso!', 'ativo': assunto.ativo})
