from flask import Blueprint, render_template, request, jsonify, flash, redirect, url_for
from flask_login import login_required, current_user
from src.models.user import db, Empresa, Segmento, FaixaFaturamento
from datetime import datetime

admin_empresa_bp = Blueprint('admin_empresa', __name__)

@admin_empresa_bp.route('/empresas')
@login_required
def listar_empresas():
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    empresas = Empresa.query.all()
    return render_template('admin/empresas/listar.html', empresas=empresas)

@admin_empresa_bp.route('/empresas/nova', methods=['GET', 'POST'])
@login_required
def nova_empresa():
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    segmentos = Segmento.query.filter_by(ativo=True).all()
    faixas_faturamento = FaixaFaturamento.query.filter_by(ativo=True).all()
    
    if request.method == 'POST':
        razao_social = request.form.get('razao_social')
        nome_fantasia = request.form.get('nome_fantasia')
        cnpj = request.form.get('cnpj')
        segmento_id = request.form.get('segmento_id')
        cidade = request.form.get('cidade')
        estado = request.form.get('estado')
        faixa_faturamento_id = request.form.get('faixa_faturamento_id')
        data_abertura = request.form.get('data_abertura')
        ativo = True if request.form.get('ativo') == 'on' else False
        
        # Validar dados
        if not all([razao_social, nome_fantasia, cnpj, segmento_id, cidade, estado, faixa_faturamento_id, data_abertura]):
            flash('Por favor, preencha todos os campos obrigatórios.', 'danger')
            return render_template('admin/empresas/form.html', segmentos=segmentos, faixas_faturamento=faixas_faturamento)
        
        # Verificar se o CNPJ já existe
        if Empresa.query.filter_by(cnpj=cnpj).first():
            flash('CNPJ já cadastrado.', 'danger')
            return render_template('admin/empresas/form.html', segmentos=segmentos, faixas_faturamento=faixas_faturamento)
        
        # Converter data_abertura para objeto Date
        try:
            data_abertura_obj = datetime.strptime(data_abertura, '%Y-%m-%d').date()
        except ValueError:
            flash('Formato de data inválido. Use o formato AAAA-MM-DD.', 'danger')
            return render_template('admin/empresas/form.html', segmentos=segmentos, faixas_faturamento=faixas_faturamento)
        
        # Criar nova empresa
        empresa = Empresa(
            razao_social=razao_social,
            nome_fantasia=nome_fantasia,
            cnpj=cnpj,
            segmento_id=segmento_id,
            cidade=cidade,
            estado=estado,
            faixa_faturamento_id=faixa_faturamento_id,
            data_abertura=data_abertura_obj,
            ativo=ativo
        )
        
        db.session.add(empresa)
        db.session.commit()
        
        flash('Empresa cadastrada com sucesso!', 'success')
        return redirect(url_for('admin_empresa.listar_empresas'))
    
    return render_template('admin/empresas/form.html', segmentos=segmentos, faixas_faturamento=faixas_faturamento)

@admin_empresa_bp.route('/empresas/editar/<int:empresa_id>', methods=['GET', 'POST'])
@login_required
def editar_empresa(empresa_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    empresa = Empresa.query.get_or_404(empresa_id)
    segmentos = Segmento.query.filter_by(ativo=True).all()
    faixas_faturamento = FaixaFaturamento.query.filter_by(ativo=True).all()
    
    if request.method == 'POST':
        razao_social = request.form.get('razao_social')
        nome_fantasia = request.form.get('nome_fantasia')
        cnpj = request.form.get('cnpj')
        segmento_id = request.form.get('segmento_id')
        cidade = request.form.get('cidade')
        estado = request.form.get('estado')
        faixa_faturamento_id = request.form.get('faixa_faturamento_id')
        data_abertura = request.form.get('data_abertura')
        ativo = True if request.form.get('ativo') == 'on' else False
        
        # Validar dados
        if not all([razao_social, nome_fantasia, cnpj, segmento_id, cidade, estado, faixa_faturamento_id, data_abertura]):
            flash('Por favor, preencha todos os campos obrigatórios.', 'danger')
            return render_template('admin/empresas/form.html', empresa=empresa, segmentos=segmentos, faixas_faturamento=faixas_faturamento)
        
        # Verificar se o CNPJ já existe (exceto para a própria empresa)
        cnpj_existente = Empresa.query.filter(Empresa.cnpj == cnpj, Empresa.id != empresa_id).first()
        if cnpj_existente:
            flash('CNPJ já cadastrado para outra empresa.', 'danger')
            return render_template('admin/empresas/form.html', empresa=empresa, segmentos=segmentos, faixas_faturamento=faixas_faturamento)
        
        # Converter data_abertura para objeto Date
        try:
            data_abertura_obj = datetime.strptime(data_abertura, '%Y-%m-%d').date()
        except ValueError:
            flash('Formato de data inválido. Use o formato AAAA-MM-DD.', 'danger')
            return render_template('admin/empresas/form.html', empresa=empresa, segmentos=segmentos, faixas_faturamento=faixas_faturamento)
        
        # Atualizar empresa
        empresa.razao_social = razao_social
        empresa.nome_fantasia = nome_fantasia
        empresa.cnpj = cnpj
        empresa.segmento_id = segmento_id
        empresa.cidade = cidade
        empresa.estado = estado
        empresa.faixa_faturamento_id = faixa_faturamento_id
        empresa.data_abertura = data_abertura_obj
        empresa.ativo = ativo
        
        db.session.commit()
        
        flash('Empresa atualizada com sucesso!', 'success')
        return redirect(url_for('admin_empresa.listar_empresas'))
    
    return render_template('admin/empresas/form.html', empresa=empresa, segmentos=segmentos, faixas_faturamento=faixas_faturamento)

@admin_empresa_bp.route('/empresas/excluir/<int:empresa_id>', methods=['POST'])
@login_required
def excluir_empresa(empresa_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return jsonify({'success': False, 'message': 'Acesso não autorizado'}), 403
    
    empresa = Empresa.query.get_or_404(empresa_id)
    
    # Verificar se a empresa possui usuários vinculados
    if empresa.usuarios:
        return jsonify({'success': False, 'message': 'Esta empresa possui usuários vinculados e não pode ser excluída.'}), 400
    
    db.session.delete(empresa)
    db.session.commit()
    
    return jsonify({'success': True, 'message': 'Empresa excluída com sucesso!'})

@admin_empresa_bp.route('/empresas/alternar-status/<int:empresa_id>', methods=['POST'])
@login_required
def alternar_status(empresa_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return jsonify({'success': False, 'message': 'Acesso não autorizado'}), 403
    
    empresa = Empresa.query.get_or_404(empresa_id)
    empresa.ativo = not empresa.ativo
    db.session.commit()
    
    status = 'ativada' if empresa.ativo else 'desativada'
    return jsonify({'success': True, 'message': f'Empresa {status} com sucesso!', 'ativo': empresa.ativo})
