from flask import Blueprint, render_template, request, jsonify, flash, redirect, url_for
from flask_login import login_required, current_user
from src.models.user import db, FaixaFaturamento

admin_faixa_bp = Blueprint('admin_faixa', __name__)

@admin_faixa_bp.route('/faixas')
@login_required
def listar_faixas():
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    faixas = FaixaFaturamento.query.all()
    return render_template('admin/faixas/listar.html', faixas=faixas)

@admin_faixa_bp.route('/faixas/nova', methods=['GET', 'POST'])
@login_required
def nova_faixa():
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    if request.method == 'POST':
        codigo = request.form.get('codigo')
        valor_de = float(request.form.get('valor_de'))
        valor_ate = float(request.form.get('valor_ate'))
        descricao = request.form.get('descricao')
        ativo = True if request.form.get('ativo') == 'on' else False
        
        # Validar dados
        if not all([codigo, descricao]) or valor_de >= valor_ate:
            flash('Por favor, preencha todos os campos corretamente. O valor inicial deve ser menor que o valor final.', 'danger')
            return render_template('admin/faixas/form.html')
        
        # Criar nova faixa
        faixa = FaixaFaturamento(
            codigo=codigo,
            valor_de=valor_de,
            valor_ate=valor_ate,
            descricao=descricao,
            ativo=ativo
        )
        
        db.session.add(faixa)
        db.session.commit()
        
        flash('Faixa de faturamento cadastrada com sucesso!', 'success')
        return redirect(url_for('admin_faixa.listar_faixas'))
    
    return render_template('admin/faixas/form.html')

@admin_faixa_bp.route('/faixas/editar/<int:faixa_id>', methods=['GET', 'POST'])
@login_required
def editar_faixa(faixa_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    faixa = FaixaFaturamento.query.get_or_404(faixa_id)
    
    if request.method == 'POST':
        codigo = request.form.get('codigo')
        valor_de = float(request.form.get('valor_de'))
        valor_ate = float(request.form.get('valor_ate'))
        descricao = request.form.get('descricao')
        ativo = True if request.form.get('ativo') == 'on' else False
        
        # Validar dados
        if not all([codigo, descricao]) or valor_de >= valor_ate:
            flash('Por favor, preencha todos os campos corretamente. O valor inicial deve ser menor que o valor final.', 'danger')
            return render_template('admin/faixas/form.html', faixa=faixa)
        
        # Atualizar faixa
        faixa.codigo = codigo
        faixa.valor_de = valor_de
        faixa.valor_ate = valor_ate
        faixa.descricao = descricao
        faixa.ativo = ativo
        
        db.session.commit()
        
        flash('Faixa de faturamento atualizada com sucesso!', 'success')
        return redirect(url_for('admin_faixa.listar_faixas'))
    
    return render_template('admin/faixas/form.html', faixa=faixa)

@admin_faixa_bp.route('/faixas/excluir/<int:faixa_id>', methods=['POST'])
@login_required
def excluir_faixa(faixa_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return jsonify({'success': False, 'message': 'Acesso não autorizado'}), 403
    
    faixa = FaixaFaturamento.query.get_or_404(faixa_id)
    
    # Verificar se a faixa está sendo usada por alguma empresa
    if faixa.empresas:
        return jsonify({'success': False, 'message': 'Esta faixa de faturamento está sendo utilizada por empresas e não pode ser excluída.'}), 400
    
    db.session.delete(faixa)
    db.session.commit()
    
    return jsonify({'success': True, 'message': 'Faixa de faturamento excluída com sucesso!'})

@admin_faixa_bp.route('/faixas/alternar-status/<int:faixa_id>', methods=['POST'])
@login_required
def alternar_status(faixa_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return jsonify({'success': False, 'message': 'Acesso não autorizado'}), 403
    
    faixa = FaixaFaturamento.query.get_or_404(faixa_id)
    faixa.ativo = not faixa.ativo
    db.session.commit()
    
    status = 'ativada' if faixa.ativo else 'desativada'
    return jsonify({'success': True, 'message': f'Faixa de faturamento {status} com sucesso!', 'ativo': faixa.ativo})
