from flask import Blueprint, render_template, request, jsonify, flash, redirect, url_for
from flask_login import login_required, current_user
from src.models.user import db, Segmento

admin_segmento_bp = Blueprint('admin_segmento', __name__)

@admin_segmento_bp.route('/segmentos')
@login_required
def listar_segmentos():
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    segmentos = Segmento.query.all()
    return render_template('admin/segmentos/listar.html', segmentos=segmentos)

@admin_segmento_bp.route('/segmentos/novo', methods=['GET', 'POST'])
@login_required
def novo_segmento():
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    if request.method == 'POST':
        codigo = request.form.get('codigo')
        nome = request.form.get('nome')
        ativo = True if request.form.get('ativo') == 'on' else False
        
        # Validar dados
        if not all([codigo, nome]):
            flash('Por favor, preencha todos os campos obrigatórios.', 'danger')
            return render_template('admin/segmentos/form.html')
        
        # Criar novo segmento
        segmento = Segmento(
            codigo=codigo,
            nome=nome,
            ativo=ativo
        )
        
        db.session.add(segmento)
        db.session.commit()
        
        flash('Segmento cadastrado com sucesso!', 'success')
        return redirect(url_for('admin_segmento.listar_segmentos'))
    
    return render_template('admin/segmentos/form.html')

@admin_segmento_bp.route('/segmentos/editar/<int:segmento_id>', methods=['GET', 'POST'])
@login_required
def editar_segmento(segmento_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    segmento = Segmento.query.get_or_404(segmento_id)
    
    if request.method == 'POST':
        codigo = request.form.get('codigo')
        nome = request.form.get('nome')
        ativo = True if request.form.get('ativo') == 'on' else False
        
        # Validar dados
        if not all([codigo, nome]):
            flash('Por favor, preencha todos os campos obrigatórios.', 'danger')
            return render_template('admin/segmentos/form.html', segmento=segmento)
        
        # Atualizar segmento
        segmento.codigo = codigo
        segmento.nome = nome
        segmento.ativo = ativo
        
        db.session.commit()
        
        flash('Segmento atualizado com sucesso!', 'success')
        return redirect(url_for('admin_segmento.listar_segmentos'))
    
    return render_template('admin/segmentos/form.html', segmento=segmento)

@admin_segmento_bp.route('/segmentos/excluir/<int:segmento_id>', methods=['POST'])
@login_required
def excluir_segmento(segmento_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return jsonify({'success': False, 'message': 'Acesso não autorizado'}), 403
    
    segmento = Segmento.query.get_or_404(segmento_id)
    
    # Verificar se o segmento está sendo usado por alguma empresa
    if segmento.empresas:
        return jsonify({'success': False, 'message': 'Este segmento está sendo utilizado por empresas e não pode ser excluído.'}), 400
    
    db.session.delete(segmento)
    db.session.commit()
    
    return jsonify({'success': True, 'message': 'Segmento excluído com sucesso!'})

@admin_segmento_bp.route('/segmentos/alternar-status/<int:segmento_id>', methods=['POST'])
@login_required
def alternar_status(segmento_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return jsonify({'success': False, 'message': 'Acesso não autorizado'}), 403
    
    segmento = Segmento.query.get_or_404(segmento_id)
    segmento.ativo = not segmento.ativo
    db.session.commit()
    
    status = 'ativado' if segmento.ativo else 'desativado'
    return jsonify({'success': True, 'message': f'Segmento {status} com sucesso!', 'ativo': segmento.ativo})
