from flask import Blueprint, render_template, request, jsonify, flash, redirect, url_for
from flask_login import login_required, current_user
from werkzeug.security import generate_password_hash
from src.models.user import db, User, Empresa

admin_usuario_bp = Blueprint('admin_usuario', __name__)

@admin_usuario_bp.route('/usuarios')
@login_required
def listar_usuarios():
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    usuarios = User.query.all()
    return render_template('admin/usuarios/listar.html', usuarios=usuarios)

@admin_usuario_bp.route('/usuarios/novo', methods=['GET', 'POST'])
@login_required
def novo_usuario():
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    empresas = Empresa.query.filter_by(ativo=True).all()
    
    if request.method == 'POST':
        nome = request.form.get('nome')
        email = request.form.get('email')
        senha = request.form.get('senha')
        confirmar_senha = request.form.get('confirmar_senha')
        tipo_perfil = request.form.get('tipo_perfil')
        empresa_id = request.form.get('empresa_id') if request.form.get('empresa_id') else None
        ativo = True if request.form.get('ativo') == 'on' else False
        
        # Validar dados
        if not all([nome, email, senha, confirmar_senha, tipo_perfil]):
            flash('Por favor, preencha todos os campos obrigatórios.', 'danger')
            return render_template('admin/usuarios/form.html', empresas=empresas)
        
        if senha != confirmar_senha:
            flash('As senhas não coincidem.', 'danger')
            return render_template('admin/usuarios/form.html', empresas=empresas)
        
        # Verificar se o email já existe
        if User.query.filter_by(email=email).first():
            flash('Email já cadastrado.', 'danger')
            return render_template('admin/usuarios/form.html', empresas=empresas)
        
        # Verificar se o tipo de perfil é válido
        if tipo_perfil not in ['administrador', 'consultor', 'cliente']:
            flash('Tipo de perfil inválido.', 'danger')
            return render_template('admin/usuarios/form.html', empresas=empresas)
        
        # Se for cliente, empresa_id é obrigatório
        if tipo_perfil == 'cliente' and not empresa_id:
            flash('Para usuários do tipo Cliente, é necessário selecionar uma empresa.', 'danger')
            return render_template('admin/usuarios/form.html', empresas=empresas)
        
        # Criar novo usuário
        usuario = User(
            nome=nome,
            email=email,
            senha_hash=generate_password_hash(senha),
            tipo_perfil=tipo_perfil,
            empresa_id=empresa_id,
            ativo=ativo
        )
        
        db.session.add(usuario)
        db.session.commit()
        
        flash('Usuário cadastrado com sucesso!', 'success')
        return redirect(url_for('admin_usuario.listar_usuarios'))
    
    return render_template('admin/usuarios/form.html', empresas=empresas)

@admin_usuario_bp.route('/usuarios/editar/<int:usuario_id>', methods=['GET', 'POST'])
@login_required
def editar_usuario(usuario_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return render_template('errors/403.html'), 403
    
    usuario = User.query.get_or_404(usuario_id)
    empresas = Empresa.query.filter_by(ativo=True).all()
    
    if request.method == 'POST':
        nome = request.form.get('nome')
        email = request.form.get('email')
        senha = request.form.get('senha')
        confirmar_senha = request.form.get('confirmar_senha')
        tipo_perfil = request.form.get('tipo_perfil')
        empresa_id = request.form.get('empresa_id') if request.form.get('empresa_id') else None
        ativo = True if request.form.get('ativo') == 'on' else False
        
        # Validar dados
        if not all([nome, email, tipo_perfil]):
            flash('Por favor, preencha todos os campos obrigatórios.', 'danger')
            return render_template('admin/usuarios/form.html', usuario=usuario, empresas=empresas)
        
        # Verificar se o email já existe (exceto para o próprio usuário)
        email_existente = User.query.filter(User.email == email, User.id != usuario_id).first()
        if email_existente:
            flash('Email já cadastrado para outro usuário.', 'danger')
            return render_template('admin/usuarios/form.html', usuario=usuario, empresas=empresas)
        
        # Verificar se o tipo de perfil é válido
        if tipo_perfil not in ['administrador', 'consultor', 'cliente']:
            flash('Tipo de perfil inválido.', 'danger')
            return render_template('admin/usuarios/form.html', usuario=usuario, empresas=empresas)
        
        # Se for cliente, empresa_id é obrigatório
        if tipo_perfil == 'cliente' and not empresa_id:
            flash('Para usuários do tipo Cliente, é necessário selecionar uma empresa.', 'danger')
            return render_template('admin/usuarios/form.html', usuario=usuario, empresas=empresas)
        
        # Atualizar usuário
        usuario.nome = nome
        usuario.email = email
        usuario.tipo_perfil = tipo_perfil
        usuario.empresa_id = empresa_id
        usuario.ativo = ativo
        
        # Atualizar senha apenas se fornecida
        if senha and confirmar_senha:
            if senha != confirmar_senha:
                flash('As senhas não coincidem.', 'danger')
                return render_template('admin/usuarios/form.html', usuario=usuario, empresas=empresas)
            
            usuario.senha_hash = generate_password_hash(senha)
        
        db.session.commit()
        
        flash('Usuário atualizado com sucesso!', 'success')
        return redirect(url_for('admin_usuario.listar_usuarios'))
    
    return render_template('admin/usuarios/form.html', usuario=usuario, empresas=empresas)

@admin_usuario_bp.route('/usuarios/excluir/<int:usuario_id>', methods=['POST'])
@login_required
def excluir_usuario(usuario_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return jsonify({'success': False, 'message': 'Acesso não autorizado'}), 403
    
    # Não permitir excluir o próprio usuário
    if usuario_id == current_user.id:
        return jsonify({'success': False, 'message': 'Não é possível excluir o próprio usuário.'}), 400
    
    usuario = User.query.get_or_404(usuario_id)
    
    db.session.delete(usuario)
    db.session.commit()
    
    return jsonify({'success': True, 'message': 'Usuário excluído com sucesso!'})

@admin_usuario_bp.route('/usuarios/alternar-status/<int:usuario_id>', methods=['POST'])
@login_required
def alternar_status(usuario_id):
    # Verificar se o usuário é administrador
    if current_user.tipo_perfil != 'administrador':
        return jsonify({'success': False, 'message': 'Acesso não autorizado'}), 403
    
    # Não permitir desativar o próprio usuário
    if usuario_id == current_user.id:
        return jsonify({'success': False, 'message': 'Não é possível alterar o status do próprio usuário.'}), 400
    
    usuario = User.query.get_or_404(usuario_id)
    usuario.ativo = not usuario.ativo
    db.session.commit()
    
    status = 'ativado' if usuario.ativo else 'desativado'
    return jsonify({'success': True, 'message': f'Usuário {status} com sucesso!', 'ativo': usuario.ativo})
