from flask import Blueprint, render_template, request, jsonify, flash, redirect, url_for
from flask_login import login_required, current_user
from src.models.user import db, Pergunta, Empresa, Resposta

consultor_pesquisa_bp = Blueprint('consultor_pesquisa', __name__)

@consultor_pesquisa_bp.route('/pesquisas')
@login_required
def listar_pesquisas():
    # Verificar se o usuário é consultor
    if current_user.tipo_perfil != 'consultor':
        return render_template('errors/403.html'), 403
    
    # Obter empresas ativas para pesquisa
    empresas = Empresa.query.filter_by(ativo=True).all()
    
    return render_template('consultor/pesquisas/listar.html', empresas=empresas)

@consultor_pesquisa_bp.route('/pesquisas/empresa/<int:empresa_id>')
@login_required
def selecionar_empresa(empresa_id):
    # Verificar se o usuário é consultor
    if current_user.tipo_perfil != 'consultor':
        return render_template('errors/403.html'), 403
    
    # Verificar se a empresa existe e está ativa
    empresa = Empresa.query.filter_by(id=empresa_id, ativo=True).first_or_404()
    
    # Obter filtros disponíveis
    setores = db.session.query(Pergunta.setor_id, Pergunta.setor).filter(Pergunta.ativo == True).distinct().all()
    assuntos = db.session.query(Pergunta.assunto_id, Pergunta.assunto).filter(Pergunta.ativo == True).distinct().all()
    atividades = db.session.query(Pergunta.atividade_id, Pergunta.atividade).filter(Pergunta.ativo == True).distinct().all()
    
    return render_template('consultor/pesquisas/filtros.html', 
                          empresa=empresa, 
                          setores=setores, 
                          assuntos=assuntos, 
                          atividades=atividades)

@consultor_pesquisa_bp.route('/pesquisas/empresa/<int:empresa_id>/perguntas', methods=['GET', 'POST'])
@login_required
def perguntas_empresa(empresa_id):
    # Verificar se o usuário é consultor
    if current_user.tipo_perfil != 'consultor':
        return render_template('errors/403.html'), 403
    
    # Verificar se a empresa existe e está ativa
    empresa = Empresa.query.filter_by(id=empresa_id, ativo=True).first_or_404()
    
    # Obter filtros da requisição
    setor_id = request.args.get('setor_id', type=int)
    assunto_id = request.args.get('assunto_id', type=int)
    atividade_id = request.args.get('atividade_id', type=int)
    
    # Construir query base
    query = Pergunta.query.filter(Pergunta.ativo == True)
    
    # Aplicar filtros se fornecidos
    if setor_id:
        query = query.filter(Pergunta.setor_id == setor_id)
    if assunto_id:
        query = query.filter(Pergunta.assunto_id == assunto_id)
    if atividade_id:
        query = query.filter(Pergunta.atividade_id == atividade_id)
    
    # Obter perguntas filtradas
    perguntas = query.all()
    
    if request.method == 'POST':
        # Processar respostas
        for pergunta in perguntas:
            resposta_valor = request.form.get(f'resposta_{pergunta.id}')
            
            if resposta_valor:
                # Verificar se já existe uma resposta para esta pergunta e empresa
                resposta_existente = Resposta.query.filter_by(
                    pergunta_id=pergunta.id,
                    empresa_id=empresa_id
                ).first()
                
                if resposta_existente:
                    # Atualizar resposta existente
                    resposta_existente.valor = resposta_valor
                    resposta_existente.usuario_id = current_user.id
                else:
                    # Criar nova resposta
                    nova_resposta = Resposta(
                        pergunta_id=pergunta.id,
                        empresa_id=empresa_id,
                        valor=resposta_valor,
                        usuario_id=current_user.id
                    )
                    db.session.add(nova_resposta)
                
        db.session.commit()
        flash('Respostas salvas com sucesso!', 'success')
        return redirect(url_for('consultor_pesquisa.listar_pesquisas'))
    
    # Obter respostas existentes para pré-preenchimento
    respostas = {}
    respostas_existentes = Resposta.query.filter_by(empresa_id=empresa_id).all()
    for resposta in respostas_existentes:
        respostas[resposta.pergunta_id] = resposta.valor
    
    return render_template('consultor/pesquisas/perguntas.html', 
                          empresa=empresa, 
                          perguntas=perguntas,
                          respostas=respostas)

@consultor_pesquisa_bp.route('/pesquisas/historico')
@login_required
def historico_pesquisas():
    # Verificar se o usuário é consultor
    if current_user.tipo_perfil != 'consultor':
        return render_template('errors/403.html'), 403
    
    # Obter empresas com respostas cadastradas pelo consultor atual
    empresas_com_respostas = db.session.query(Empresa).\
        join(Resposta, Resposta.empresa_id == Empresa.id).\
        filter(Resposta.usuario_id == current_user.id).\
        distinct().all()
    
    return render_template('consultor/pesquisas/historico.html', empresas=empresas_com_respostas)
